/*---------------------------------------------------------------------------*\

    FILE....: OBJTRACK.CPP
    TYPE....: C++ Module
    AUTHOR..: David Rowe
    DATE....: 25/3/98

    This module maintains a database of the VPB configuration manager
    objects.  This database is used to determine a device handle 
    (channel) from an object and board, or vice versa.
	 
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "wobbly.h"
#include "objtrack.h"
#include "vpbapi.h"
#include "apifunc.h"

/*---------------------------------------------------------------------------*\

				       DEFINES

\*--------------------------------------------------------------------------*/

typedef struct OBJS {
	USHORT	cls;		// class of object
	USHORT	id;		// id of object on this board
	USHORT	handle;		// handle of channel
	USHORT	b;		// board number
	struct	OBJS *next;	// next object in linked list	
} OBJ;

#define V12PCI_PORTS  12

/*---------------------------------------------------------------------------*\

				   STATICS

\*--------------------------------------------------------------------------*/

static OBJ *start;		// first object in linked list
static OBJ *end;		// last object in linked list

/*---------------------------------------------------------------------------*\

				  FUNCTIONS

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

	FUNCTION: objtrack_open
	AUTHOR..: David Rowe
	DATE....: 25/3/98

	Initialises the objtrack module.

\*--------------------------------------------------------------------------*/

void objtrack_open() {
	start = NULL;	
	end = NULL;
}

/*---------------------------------------------------------------------------*\

	FUNCTION: objtrack_close
	AUTHOR..: David Rowe
	DATE....: 19/3/98

	Closes the playrec module.

\*--------------------------------------------------------------------------*/

void objtrack_close() {
	OBJ	*po = start;
	OBJ	*d;

	// delete all objects in linked list

	while(po != NULL) {
		d = po;
		po = po->next;
		delete d;
	}
		
}

/*---------------------------------------------------------------------------*\

	FUNCTION: objtrack_add_object()
	AUTHOR..: David Rowe
	DATE....: 25/3/98

	Adds a config manager object to the database.

\*--------------------------------------------------------------------------*/

void objtrack_add_object(USHORT cls, USHORT id, USHORT handle, USHORT b)
//	USHORT	cls			class of object
//	USHORT	id			id of object on this board
//	USHORT	handle		        handle of channel
//	USHORT	b			board number
{
	OBJ	*po;

	po = new OBJ;
	po->cls = cls;
	po->id = id;
	po->handle = handle;
	po->b = b;
	po->id = id;
	po->next = NULL;

	// attach new object to end of list

	if (start == NULL) {
		start = end = po;
	}
	else {
		end->next = po;
		end = po;
	}
}

/*---------------------------------------------------------------------------*\

	FUNCTION: objtrack_handle_to_id()
	AUTHOR..: David Rowe
	DATE....: 25/3/98

	Finds the handle of an object given the id.

\*--------------------------------------------------------------------------*/

int objtrack_id_to_handle(USHORT cls, USHORT id, USHORT b)
//	USHORT	cls			class of object
//	USHORT	id			id of object on this board
//	USHORT	b			board number
{
	OBJ	*po = start;

	// V12PCI has a different mapping

        if (vpb_c->vpbreg(0)->model == VPB_V12PCI) {
		return(id + b*V12PCI_PORTS);
	}

	while(po != NULL) {
		if ((po->cls == cls) && (po->id == id) && (po->b == b))
			return(po->handle);
		po = po->next;
	}

	assert(0);
	return(0);
}

/*---------------------------------------------------------------------------*\

	FUNCTION: objtrack_handle_to_id()
	AUTHOR..: David Rowe
	DATE....: 25/3/98

	Finds the handle of an object given the class and channel handle.  If
	there is more than one object of this class on this channel this 
	function will only return the id of the first one found.

\*--------------------------------------------------------------------------*/

int objtrack_handle_to_id(USHORT cls, USHORT handle)
//	USHORT	cls		class of object
//	USHORT	handle		handle of channel
{
	OBJ	*po = start;

	// V12PCI has a different mapping

        if (vpb_c->vpbreg(0)->model == VPB_V12PCI) {
		return(handle % V12PCI_PORTS);
	}

	while(po != NULL) {
		if ((po->cls == cls) && (po->handle == handle))
			return(po->id);
		po = po->next;
	}

	assert(0);
	return(0);
}

